/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.j2ee.clientproject;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.platform.JavaPlatform;
import org.netbeans.modules.java.platform.implspi.JavaPlatformProvider;
import org.netbeans.api.java.platform.Specification;
import org.netbeans.spi.java.classpath.support.ClassPathSupport;
import org.openide.filesystems.FileObject;
import org.openide.modules.SpecificationVersion;

/**
 * @author Tomas Zezula
 */
public final class TestPlatformProvider implements JavaPlatformProvider {
    
    private JavaPlatform defaultPlatform;
    private JavaPlatform explicitPlatform;
    private PropertyChangeSupport support;
    private boolean hideExplicitPlatform;
    
    /** Default constructor for lookup. */
    public TestPlatformProvider() {
        this(null, null);
    }
    
    public TestPlatformProvider(ClassPath defaultPlatformBootClassPath, ClassPath explicitPlatformBootClassPath) {
        this.support = new PropertyChangeSupport(this);
        this.defaultPlatform = new TestPlatform("default_platform", "1.5", defaultPlatformBootClassPath);
        this.explicitPlatform = new TestPlatform("ExplicitPlatform", "1.4", explicitPlatformBootClassPath);
    }
    
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        this.support.removePropertyChangeListener(listener);
    }
    
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        this.support.addPropertyChangeListener(listener);
    }
    
    public JavaPlatform[] getInstalledPlatforms()  {
        if (this.hideExplicitPlatform) {
            return new JavaPlatform[] {
                this.defaultPlatform,
            };
        } else {
            return new JavaPlatform[] {
                this.defaultPlatform,
                this.explicitPlatform,
            };
        }
    }
    
    public JavaPlatform getDefaultPlatform() {
        return this.defaultPlatform;
    }
    
    public void setExplicitPlatformVisible(boolean value) {
        this.hideExplicitPlatform = !value;
        this.support.firePropertyChange(PROP_INSTALLED_PLATFORMS,null,null);
    }
    
    private static class TestPlatform extends JavaPlatform {
        
        private final String systemName;
        private final Map<String, String> properties;
        private final ClassPath bootClassPath;
        private final String specVersion;
        
        public TestPlatform(String systemName, String specVersion, ClassPath bootCP) {
            this.systemName = systemName;
            this.specVersion = specVersion;
            if (bootCP == null) {
                try {
                    this.bootClassPath = ClassPathSupport.createClassPath(new URL[] {
                        // This file does not really have to exist - just needs to have a well-known location.
                        // Cf. ClasspathsTest.
                        new URL("jar:file:/c:/java/" + systemName + "/jre/lib/rt.jar!/"),
                    });
                } catch (MalformedURLException e) {
                    throw new AssertionError(e);
                }
            } else {
                this.bootClassPath = bootCP;
            }
            this.properties = Collections.singletonMap("platform.ant.name", this.systemName);
        }
        
        public FileObject findTool(String toolName) {
            return null;
        }
        
        public String getVendor() {
            return "me";
        }
        
        public ClassPath getStandardLibraries() {
            return null;
        }
        
        public Specification getSpecification() {
            return new Specification("j2se", new SpecificationVersion(specVersion));
        }
        
        public ClassPath getSourceFolders() {
            return null;
        }
        
        public Map<String, String> getProperties() {
            return this.properties;
        }
        
        public List<URL> getJavadocFolders() {
            return null;
        }
        
        public Collection<FileObject> getInstallFolders() {
            return null;
        }
        
        public String getDisplayName() {
            return this.systemName;
        }
        
        public ClassPath getBootstrapLibraries() {
            return this.bootClassPath;
        }
    }
}
