// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_WEBUI_SETTINGS_CHROMEOS_SEARCH_SEARCH_CONCEPT_H_
#define CHROME_BROWSER_UI_WEBUI_SETTINGS_CHROMEOS_SEARCH_SEARCH_CONCEPT_H_

#include "chrome/browser/ui/webui/settings/chromeos/search/search_result_icon.mojom.h"

namespace chromeos {
namespace settings {

// Represents a potential search result. In this context, "concept" refers to
// the fact that this search result represents an idea which may be described
// by more than just one phrase. For example, a concept of "Display settings"
// may also be described as "Monitor settings".
//
// Each concept has a canonical description search tag as well as up to
// |kMaxAltTagsPerConcept| alternate descriptions search tags.
struct SearchConcept {
  static constexpr size_t kMaxAltTagsPerConcept = 4;
  static constexpr int kAltTagEnd = 0;

  SearchConcept(const SearchConcept& other) = default;
  SearchConcept& operator=(const SearchConcept& other) = default;

  // Message ID (from os_settings_search_tag_strings.grdp) corresponding to the
  // canonical search tag for this concept.
  int canonical_message_id;

  // URL path corresponding to the settings subpage at which the user can
  // change a setting associated with the tag. This string can also contain
  // URL parameters.
  //
  // Example 1 - Display settings (chrome://os-settings/device/display):
  //             ==> "device/display".
  // Example 2 - Wi-Fi settings (chrome://os-settings/networks?type=WiFi):
  //             ==> "networks?type=WiFi"
  const char* url_path_with_parameters;

  // Icon to display for search results associated with this concept.
  mojom::SearchResultIcon icon;

  // Alternate message IDs (from os_settings_search_tag_strings.grdp)
  // corresponding to this concept. These IDs refer to messages which represent
  // an alternate way of describing the same concept (e.g., "Monitor settings"
  // is an alternate phrase for "Display settings").
  //
  // This field provides up to |kMaxAltTagsPerConcept| alternate tags, but not
  // all concepts will require this many. A value of kAltTagEnd is used to
  // indicate that there are no further tags.
  //
  // Example 1 - Four alternate tags: [1234, 1235, 1236, 1237]
  // Example 2 - Two alternate tags: [1234, 1235, kAltTagEnd, _]
  // Example 3 - Zero alternate tags: [kAltTagEnd, _, _, _]
  int alt_tag_ids[kMaxAltTagsPerConcept] = {kAltTagEnd};
};

}  // namespace settings
}  // namespace chromeos

#endif  // CHROME_BROWSER_UI_WEBUI_SETTINGS_CHROMEOS_SEARCH_SEARCH_CONCEPT_H_
