#-*- coding: utf-8 -*-

# Copyright 2010-2013 Calculate Ltd. http://www.calculate-linux.org
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

import sys
from calculate.core.server.func import Action, Tasks
from calculate.lib.cl_lang import setLocalTranslate, getLazyLocalTranslate
from calculate.lib.utils.colortext import get_color_print, Colors
from calculate.lib.utils.files import FilesError
from calculate.update.update import UpdateError
from calculate.update.emerge_parser import EmergeError
from calculate.update.package_tools import GitError, Eix, EmergeLog, \
    EmergeLogNamedTask, PackageList

setLocalTranslate('cl_update3', sys.modules[__name__])
__ = getLazyLocalTranslate(_)


class ClUpdateAction(Action):
    """
    Действие обновление конфигурационных файлов
    """
    # ошибки, которые отображаются без подробностей
    native_error = (FilesError, UpdateError, GitError, EmergeError)

    successMessage = None
    failedMessage = None
    interruptMessage = __("Update manually interrupted")

    def was_installed(pkg, task_name):
        def func():
            task = EmergeLog(EmergeLogNamedTask(task_name))
            return bool(PackageList(task.list)[pkg])
        return func

    def need_upgrade(pkg):
        def func():
            return bool(Eix(pkg, Eix.Option.Upgrade).get_packages())
        return func

    def pkg_color(text):
        return text

    log_names = {'schedule': "schedule",
                 'premerge': "check updates",
                 'python_updater': "update python modules",
                 'perl_cleaner': "update perl modules",
                 'kernel_modules': "update kernel modules",
                 'depclean': "depclean",
                 'xorg_modules': "update xorg modules",
                 'preserved_libs': "update preserved libs",
                 'revdep': "revdep rebuild"}

    emerge_tasks = [
        {'name': 'premerge_group',
         'group': __("Checking for updates"),
         'tasks': [
            {'name': 'premerge',
             'message': __("Calculating dependencies"),
             'method': 'Update.premerge("-uDN","--with-bdeps=y","@world")',
             'condition': lambda Get:Get('cl_update_sync_only_set') == 'off'
            }],
        },
        {'name': 'premerge:update',
         'condition': lambda Get:Get('cl_update_pretend_set') == 'off',
         'depend': Tasks.result("premerge", eq='yes')
        },
        {'name': 'update_other',
         'condition': lambda Get: ( Get('cl_update_pretend_set') == 'off' and
                                    Get('cl_update_sync_only_set') == 'off')
        },
        {'name': 'update:update_portage',
         'group': __("Updating Portage"),
         'tasks': [
             {'name': 'update:update_portage_pkg',
              'message': __("Updating {0}").format(
                  pkg_color("sys-apps/portage")),
              'method': 'Update.emerge("-u","portage")',
              'condition': need_upgrade('sys-apps/portage$')
             },
         ]
        },
        {'name': 'update:update_python',
         'group': __("Updating Python"),
         'tasks': [
             {'name': 'update:update_python_pkg',
              'message': __('Updating {0}').format(
                  pkg_color('dev-lang/python')),
              'method': 'Update.emerge("-u","dev-lang/python")',
              'condition': need_upgrade('dev-lang/python$')
             },
             {'name': 'update:python_updater',
              'message': __('Find & rebuild packages broken due '
                            'to a Python upgrade'),
              'method': 'Update.emergelike("python-updater")',
              'condition': was_installed('dev-lang/python$',
                                         log_names['python_updater']),
              'decoration': 'Update.update_task("%s")' % log_names[
                  'python_updater']
             },
         ]
        },
        {'name': 'update:update_perl',
         'group': __("Updating Perl"),
         'tasks': [
             {'name': 'update:update_perl_pkg',
              'message': __('Updating {0}').format(pkg_color('dev-lang/perl')),
              'method': 'Update.emerge("-u","dev-lang/perl")',
              'condition': need_upgrade('dev-lang/perl$')
             },
             {'name': 'update:perl_cleaner',
              'message': __('Find & rebuild packages and Perl header files '
                            'broken due to a perl upgrade'),
              'method': 'Update.emergelike("perl-cleaner", "all")',
              'condition': was_installed('dev-lang/perl$',
                                         log_names['perl_cleaner']),
              'decoration': 'Update.update_task("%s")' % log_names[
                  'perl_cleaner']
             },
         ]
        },
        {'name': 'update:update_world',
         'group': __("Updating packages"),
         'tasks': [
             {'name': 'update:update_world',
              'message': __("Calculating dependencies"),
              'method': 'Update.emerge("-uDN","--with-bdeps=y","@world")',
             }
         ]
        },
        {'name': 'update_other:depclean',
         'group': __("Cleaning the system from needless packages"),
         'tasks': [
             {'name': 'update_other:update_depclean',
              'message': __("Calculating dependencies"),
              'method': 'Update.depclean()',
              'condition': was_installed('.*', log_names['depclean']),
              'decoration': 'Update.update_task("%s")' % log_names['depclean']
             },
         ]
        },
        {'name': 'update_other:update_modules',
         'group': __("Rebuilding dependent modules"),
         'tasks': [
             {'name': 'update_other:module_rebuild',
              'message': __('Updating Kernel modules'),
              'method': 'Update.emerge("@module-rebuild")',
              'condition': was_installed('sys-kernel/.*source',
                                         log_names['kernel_modules']),
              'decoration': 'Update.update_task("%s")' % log_names[
                  'kernel_modules']
             },
             {'name': 'update_other:x11_module_rebuild',
              'message': __('Updating X.Org server modules'),
              'method': 'Update.emerge("@x11-module-rebuild")',
              'condition': was_installed('x11-base/xorg-server',
                                         log_names['xorg_modules']),
              'decoration': 'Update.update_task("%s")' % log_names[
                  'xorg_modules']
             },
             {'name': 'update_other:preserved_rebuild',
              'message': __('Updating preserved libraries'),
              'method': 'Update.emerge("@preserved-rebuild")',
              'condition': was_installed('.*', log_names['preserved_libs']),
              'decoration': 'Update.update_task("%s")' % log_names[
                  'preserved_libs']
             },
             {'name': 'update_other:revdev_rebuild',
              'message': __('Checking reverse dependencies'),
              'method': 'Update.revdep_rebuild("revdep-rebuild")',
              'condition': was_installed('.*', log_names['revdep']),
              'decoration': 'Update.update_task("%s")' % log_names['revdep']
             },
             {'name': 'update_other:dispatch_conf_end',
              'message': __("Updating configuration files"),
              'method':'Update.dispatchConf()',
              'condition': lambda Get: ((Get('cl_update_rev_set') == 'on' or
                                         Get('cl_rebuild_world_set') == 'on') and
                                        Get('cl_update_pretend_set') == 'off')
             },
         ]
        },
        {'name': 'update:set_upto_date_cache',
         'method': 'Update.setUpToDateCache()'
        }
    ]

    # список задач для дейсвия
    tasks = [
        {'name': 'check_schedule',
         'method': 'Update.checkSchedule(cl_update_autocheck_interval,'
                   'cl_update_autocheck_set)',
         'condition': lambda Get: (
             Get('cl_update_autocheck_schedule_set') == 'on'),
         },
        {'name': 'check_run',
         'method': 'Update.checkRun(cl_update_wait_another_set)'},
        {'name': 'reps_synchronization',
         'group': __("Repositories synchronization"),
         'tasks': [
             {'name': 'sync_reps',
              'foreach': 'cl_update_sync_rep',
              'message': __("Syncing the {eachvar:capitalize} repository"),
              'method': 'Update.syncRepositories(eachvar)',
              'condition': lambda Get: Get('cl_update_sync_rep')
             },
             {'name': 'sync_other_reps',
              'foreach': 'cl_update_other_rep_name',
              'message': __("Syncing the {eachvar:capitalize} repository"),
              'method': 'Update.syncLaymanRepository(eachvar)',
              'condition': lambda Get: Get('cl_update_other_set') == 'on'
             },
             {'name': 'sync_reps:regen_cache',
              'foreach': 'cl_update_sync_overlay_rep',
              'essential': False,
              'method': 'Update.regenCache(eachvar)',
              'condition': (
                  lambda Get: (Get('cl_update_outdate_set') == 'on' and
                               Get('cl_update_egencache_force') != 'skip' or
                               Get('cl_update_egencache_force') == 'force'))
             },
             {'name': 'sync_other_reps:regen_other_cache',
              'foreach': 'cl_update_other_rep_name',
              'method': 'Update.regenCache(eachvar)',
              'essential': False,
             },
             {'name': 'emerge_metadata',
              'message': __("Metadata transfer"),
              'method': 'Update.emergeMetadata()',
              'condition': (
                  lambda Get: (Get('cl_update_outdate_set') == 'on' and
                               Get('cl_update_metadata_force') != 'skip' or
                               Get('cl_update_metadata_force') == 'force'))
             },
             {'name': 'eix_update',
              'message': __("Updating the eix cache"),
              'method': 'Update.eixUpdate()',
              'condition': (
                  lambda Get: (Get('cl_update_outdate_set') == 'on' and
                               Get('cl_update_eixupdate_force') != 'skip' or
                               Get('cl_update_eixupdate_force') == 'force'))
             },
             # сообщение удачного завершения при обновлении репозиториев
             {'name': 'success_syncrep',
              'message': __("Synchronization finished"),
              'depend': (Tasks.success() & Tasks.has_any("sync_reps",
                                                         "sync_other_reps",
                                                         "emerge_metadata",
                                                         "eix_update")),
             }
         ]
        },
        {'name': 'reps_synchronization',
         'group': __("System configuration"),
         'tasks': [
            {'name': 'revision',
             'message': __("Fixing the settings"),
             'method': 'Update.applyTemplates(install.cl_source,'
                       'cl_template_clt_set,True,None,False)',
             'condition': lambda Get: (Get('cl_update_rev_set') == 'on' or
                                       Get('cl_rebuild_world_set') == 'on')
            },
            {'name': 'dispatch_conf',
             'message': __("Updating configuration files"),
             'method':'Update.dispatchConf()',
             'condition': lambda Get: ((Get('cl_update_rev_set') == 'on' or
                                       Get('cl_rebuild_world_set') == 'on') and
                                       Get('cl_update_pretend_set') == 'off')
            },
         ]
        }
    ] + emerge_tasks + [
        {'name':'failed',
         'error':__("Update failed"),
         'depend': (Tasks.failed() & Tasks.hasnot("interrupt") &
                    (Tasks.hasnot("check_schedule") |
                     Tasks.success_all("check_schedule")))},
        {'name': 'failed',
         'depend': Tasks.failed_all("check_schedule")
        },
        # сообщение удачного завершения при обновлении ревизии
        {'name': 'success_rev',
         'message': __("System update finished!"),
         'condition': lambda Get: (Get('cl_update_rev_set') == 'on' and
                                    Get('cl_update_pretend_set') == 'off')
        },
        # сообщение удачного завершения при пересоздании world
        {'name': 'success_world',
         'message': __("World rebuild finished!"),
         'condition': lambda Get: Get('cl_rebuild_world_set') == 'on'
        },
    ]
