
/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__GATEWAYSEMAPHORE_HPP)
#include <GatewaySemaphore.hpp>
#endif

#include <sys/sem.h>

namespace corelinux
{

   //
   // Default constructor not allowed
   //

   GatewaySemaphore::GatewaySemaphore( void ) throw(Assertion)
      :
      Semaphore(),
      theMaxCount(0)
   {
      NEVER_GET_HERE;
   }

   //
   // Copy constructor not allowed
   //

   GatewaySemaphore::GatewaySemaphore( GatewaySemaphoreCref ) 
                        throw( Assertion )
      :
      Semaphore(),
      theMaxCount(0)
   {
      NEVER_GET_HERE;
   }

   //
   // Default Constructor
   //

   GatewaySemaphore::GatewaySemaphore
      ( 
         SemaphoreGroupPtr aGroup,
         SemaphoreIdentifierRef aIdentifier,
         Count aValue,
         bool  aRecursionFlag,
         bool  aBalkingFlag
      ) throw (NullPointerException)
      :
      Semaphore(aGroup,aIdentifier,aRecursionFlag,aBalkingFlag),
      theMaxCount(aValue)
   {
      REQUIRE( theMaxCount > 0 );

      setValue(theMaxCount);
      theClients.clear();
   }

   //
   // Destructor
   //

   GatewaySemaphore::~GatewaySemaphore( void )
   {
      theClients.clear();
   }

   //
   // Returns true if lock value == 0
   //

   bool  GatewaySemaphore::isLocked( void )
   {
      return ( theClients.size() == theMaxCount );
   }

   //
   // Determines if thread owns one of the 
   // resources of this Gateway
   //

   bool  GatewaySemaphore::isAnOwner( void )
   {
      bool  isOwner( false );
      GUARD;

      if( theClients.size() != 0 &&
            (
               theClients.find( Thread::getThreadIdentifier() ) != 
                  theClients.end()
            ) 
         )
      {
         isOwner = true;
      }
      else
      {
         ;  // do nothing
      }

      return isOwner;
   }

   //
   // Gets the recursion depth  for the calling
   // thread. If recursion is disabled, this means the
   // count of entries into the resources, otherwise
   // it means recursion
   //

   Counter  GatewaySemaphore::getOwnerRecursionQueueLength( void )
   {
      Counter  aCount(-1);

      if( this->isAnOwner() == true )
      {
         GUARD;
         aCount = (*(theClients.find(Thread::getThreadIdentifier()))).second;
      }
      else
      {
         ;  // do nothing
      }

      return aCount;
   }

   //
   // Call for lock with wait disposition
   //

   SemaphoreOperationStatus GatewaySemaphore::lockWithWait( void )
      throw(SemaphoreException)
   {
      Guard                      myGuard(this->access());
      SemaphoreOperationStatus   aStatus( SUCCESS );
      ThreadIdentifier           aTid( Thread::getThreadIdentifier() );
      bool                       recurs( Semaphore::isRecursionEnabled() );

      //
      // If there is resource available
      //

      if( theClients.size() < theMaxCount &&
          AbstractSemaphore::getValue() != 0 )
      {
         // If first take the quick route

         if( recurs == false || theClients.size() == 0 )
         {
            myGuard.release();
            aStatus = this->lockAndAdd(aTid);
         }

         // If not first, but potential for recursion exists

         else
         {
            GatewayClientIterator   aGItr( theClients.find(aTid) );

            // if we exist, then just bump the counter

            if(  aGItr != theClients.end() )
            {
               (*aGItr).second++;

               // if we are the last owner

               if( aTid == getOwnerId() )
               {
                  ++(*this);
               }
               else
               {
                  ;  // do nothing
               }
            }

            //
            // we don't exist, so lock
            //

            else
            {
               myGuard.release();
               aStatus = this->lockAndAdd(aTid);
            }
         }
      }

      //
      // Here we are out of resources
      //

      else
      {
         GatewayClientIterator   aGItr( theClients.find(aTid) );

         // If we are recursive, and we are in the set already

         if( recurs == true )
         {
            if( aGItr != theClients.end() )
            {
               (*aGItr).second++;

               // if we are the last owner

               if( aTid == getOwnerId() )
               {
                  ++(*this);
               }
               else
               {
                  ;  // do nothing
               }
            }

            // not in set, check balking or wait

            else
            {
               if( Semaphore::isBalkingEnabled() == true )
               {
                  aStatus = BALKED;
               }
               else
               {
                  myGuard.release();
                  aStatus = this->lockAndAdd( aTid );
               }
            }
         }

         // Full house, recursion disabled, check balking,
         // or wait

         else
         {
            if( Semaphore::isBalkingEnabled() == true )
            {
               aStatus = BALKED;
            }
            else
            {
               myGuard.release();
               aStatus = this->lockAndAdd( aTid );
            }
         }
      }

      return aStatus;
   }

   //
   // Call for lock without waiting
   //

   SemaphoreOperationStatus GatewaySemaphore::lockWithNoWait( void )
      throw(SemaphoreException)
   {
      Guard                      myGuard(this->access());
      SemaphoreOperationStatus   aStatus( SUCCESS );
      ThreadIdentifier           aTid( Thread::getThreadIdentifier() );
      bool                       recurs( Semaphore::isRecursionEnabled() );

      //
      // If there is resource available
      //

      if( theClients.size() < theMaxCount &&
          AbstractSemaphore::getValue() != 0 )
      {
         // If first take the quick route

         if( recurs == false || theClients.size() == 0 )
         {
            myGuard.release();
            aStatus = this->lockAndAdd(aTid);
         }

         // If not first, but potential for recursion exists

         else
         {
            GatewayClientIterator   aGItr( theClients.find(aTid) );

            // if we exist, then just bump the counter

            if(  aGItr != theClients.end() )
            {
               (*aGItr).second++;

               // if we are the last owner

               if( aTid == getOwnerId() )
               {
                  ++(*this);
               }
               else
               {
                  ;  // do nothing
               }
            }

            //
            // we don't exist, so lock
            //

            else
            {
               myGuard.release();
               aStatus = this->lockAndAdd(aTid);
            }
         }
      }

      //
      // Here we are fresh out of resources
      //

      else
      {
         GatewayClientIterator   aGItr( theClients.find(aTid) );

         // If we are recursive, and we are in the set already

         if( recurs == true )
         {
            if( aGItr != theClients.end() )
            {
               (*aGItr).second++;

               // if we are the last owner

               if( aTid == getOwnerId() )
               {
                  ++(*this);
               }
               else
               {
                  ;  // do nothing
               }
            }

            // not in set, check balking or wait

            else
            {
               if( Semaphore::isBalkingEnabled() == true )
               {
                  aStatus = BALKED;
               }
               else
               {
                  myGuard.release();
                  aStatus = this->lockAndAdd( aTid, IPC_NOWAIT );
               }
            }
         }

         // Full house, recursion disabled, check balking,
         // or wait

         else
         {
            if( Semaphore::isBalkingEnabled() == true )
            {
               aStatus = BALKED;
            }
            else
            {
               myGuard.release();
               aStatus = this->lockAndAdd( aTid, IPC_NOWAIT );
            }
         }
      }

      return aStatus;
   }

   //
   // Call release for lock
   //

   SemaphoreOperationStatus GatewaySemaphore::release( void )
      throw( SemaphoreException )
   {
      GUARD;
      SemaphoreOperationStatus   aStatus( SUCCESS );
      ThreadIdentifier           aTid( Thread::getThreadIdentifier() );
      GatewayClientIterator      aGItr( theClients.find(aTid) );

      if( aGItr != theClients.end() )
      {
         Count aC = (*aGItr).second;

         //
         // If we are at the last, or we are not recursive if
         // we are not at the last, we need to release a 
         // resource
         //

         if( aC == 1 || Semaphore::isRecursionEnabled() == false )
         {
            if( ( aStatus = AbstractSemaphore::setUnlock( IPC_NOWAIT ) )
                == SUCCESS )
            {
               if( aTid == getOwnerId() )
               {
                  Semaphore::resetOwnerId();
               }
               else
               {
                  ;  // do nothing
               }

               if( aC == 1 )
               {
                  theClients.erase(aGItr);
               }
               else
               {
                  (*aGItr).second--;
               }
            }
            else
            {
               ;  // do nothing, let error return
            }
         }

         //
         // We are here because we have more than one length
         // and recursion is true
         //

         else
         {
            (*aGItr).second--;
         }
      }
      else
      {
         aStatus = UNAVAILABLE;
      }

      return aStatus;
   }

   //
   // Basic lock function
   //

   SemaphoreOperationStatus GatewaySemaphore::lockAndAdd
      ( 
         ThreadIdentifierRef aTid, Int aFlag 
      )
   {
      SemaphoreOperationStatus   aStatus( SUCCESS );

      //
      // Wait on the semaphore or not, depending on 
      // flag
      //

      if( (aStatus = AbstractSemaphore::setLock( aFlag ) ) == SUCCESS )
      {
         // If we are the last and not the same

         if( aTid != getOwnerId() )
         {
            Semaphore::setRecursionQueueLength(0);
         }
         else
         {
            ;  // do nothing
         }

         Semaphore::setOwnerId();

         //
         // If we exist then increment the number of resources
         //

         GatewayClientIterator   aGItr( theClients.find(aTid) );
         if( aGItr != theClients.end() )
         {
            (*aGItr).second++;
         }

         //
         // Otherwise put us in the running
         //

         else
         {
            theClients[aTid] = 1;  
         }
      }
      else
      {
         ;  // do nothing
      }
      return aStatus;
   }
}

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.5 $
   $Date: 2000/06/03 03:08:33 $
   $Locker:  $
*/



